function [x,fval,exitflag,output] = my_fminsearch(funfcn,x,options,varargin)
% ------------------------------------------------------------
% MY_FMINSEARCH 
% This is a slight modification of the 
% FMINSEARCH Multidimensional unconstrained nonlinear minimization (Nelder-Mead)
% to display the output of the nonlinear iterations.
%
% Type 'help fminsearch' for the usage of this function
% ------------------------------------------------------------

% This file is part of the CTRW toolbox v4.0
% For the usage of this file see Legal_notice.m
% Author   : Andrea Cortis
% $Original version : 26-Jun-2004$
% e-mail   : brian.berkowitz@weizmann.ac.il
%           (see http://weizmann.ac.il/ESER/people/Brian/CTRW)

% ------------------------------------------------------------

defaultopt = struct('Display','notify','MaxIter','200*numberOfVariables',...
    'MaxFunEvals','200*numberOfVariables','TolX',1e-4,'TolFun',1e-4);

% If just 'defaults' passed in, return the default options in X
if nargin==1 & nargout <= 1 & isequal(funfcn,'defaults')
    x = defaultopt;
    return
end

if nargin < 2, 
    error('FMINSEARCH requires at least two input arguments'); 
end

if nargin<3, options = []; end
n = prod(size(x));
numberOfVariables = n;

printtype = optimget(options,'Display',defaultopt,'fast');
tolx = optimget(options,'TolX',defaultopt,'fast');
tolf = optimget(options,'TolFun',defaultopt,'fast');
maxfun = optimget(options,'MaxFunEvals',defaultopt,'fast');
maxiter = optimget(options,'MaxIter',defaultopt,'fast');

% In case the defaults were gathered from calling: optimset('fminsearch'):
if ischar(maxfun)
    if isequal(lower(maxfun),'200*numberofvariables')
        maxfun = 200*numberOfVariables;
    else
        error('Option ''MaxFunEvals'' must be an integer value if not the default.')
    end
end
if ischar(maxiter)
    if isequal(lower(maxiter),'200*numberofvariables')
        maxiter = 200*numberOfVariables;
    else
        error('Option ''MaxIter'' must be an integer value if not the default.')
    end
end

switch printtype
case 'notify'
    prnt = 1;
case {'none','off'}
    prnt = 0;
case 'iter'
    prnt = 3;
case 'final'
    prnt = 2;
case 'simplex'
    prnt = 4;
otherwise
    prnt = 1;
end

header = ' Iteration   Func-count     min f(x)         Procedure';

% Convert to inline function as needed.
funfcn = fcnchk(funfcn,length(varargin));

n = prod(size(x));

% Initialize parameters
rho = 1; chi = 2; psi = 0.5; sigma = 0.5;
onesn = ones(1,n);
two2np1 = 2:n+1;
one2n = 1:n;

% Set up a simplex near the initial guess.
xin = x(:); % Force xin to be a column vector
v = zeros(n,n+1); fv = zeros(1,n+1);
v(:,1) = xin;    % Place input guess in the simplex! (credit L.Pfeffer at Stanford)
x(:) = xin;    % Change x to the form expected by funfcn 
fv(:,1) = feval(funfcn,x,varargin{:}); 

% Following improvement suggested by L.Pfeffer at Stanford
usual_delta = 0.05;             % 5 percent deltas for non-zero terms
zero_term_delta = 0.00025;      % Even smaller delta for zero elements of x
for j = 1:n
    y = xin;
    if y(j) ~= 0
        y(j) = (1 + usual_delta)*y(j);
    else 
        y(j) = zero_term_delta;
    end  
    v(:,j+1) = y;
    x(:) = y; f = feval(funfcn,x,varargin{:});
    fv(1,j+1) = f;
end     

% sort so v(1,:) has the lowest function value 
[fv,j] = sort(fv);
v = v(:,j);

how = 'initial';
itercount = 1;
func_evals = n+1;
if prnt == 3
    disp(' ')
    disp(header)
    fprintf(' %5.0f        %5.0f     %6.4e      %20s   ', itercount, func_evals, fv(1), how)
    fprintf('  x = [ ');for jj=1:length(x); fprintf(' %6.4e  ',x(jj));end;fprintf(' ]\n')
    
elseif prnt == 4
    clc
    formatsave = get(0,{'format','formatspacing'});
    format compact
    format short e
    disp(' ')
    disp(how)
    v
    fv
    func_evals
end
exitflag = 1;

% Main algorithm
% Iterate until the diameter of the simplex is less than tolx
%   AND the function values differ from the min by less than tolf,
%   or the max function evaluations are exceeded. (Cannot use OR instead of AND.)
while func_evals < maxfun & itercount < maxiter
    if max(max(abs(v(:,two2np1)-v(:,onesn)))) <= tolx & ...
            max(abs(fv(1)-fv(two2np1))) <= tolf
        break
    end
    how = '';
    
    % Compute the reflection point
    
    % xbar = average of the n (NOT n+1) best points
    xbar = sum(v(:,one2n), 2)/n;
    xr = (1 + rho)*xbar - rho*v(:,end);
    x(:) = xr; fxr = feval(funfcn,x,varargin{:});
    func_evals = func_evals+1;
    
    if fxr < fv(:,1)
        % Calculate the expansion point
        xe = (1 + rho*chi)*xbar - rho*chi*v(:,end);
        x(:) = xe; fxe = feval(funfcn,x,varargin{:});
        func_evals = func_evals+1;
        if fxe < fxr
            v(:,end) = xe;
            fv(:,end) = fxe;
            how = 'expand';
        else
            v(:,end) = xr; 
            fv(:,end) = fxr;
            how = 'reflect';
        end
    else % fv(:,1) <= fxr
        if fxr < fv(:,n)
            v(:,end) = xr; 
            fv(:,end) = fxr;
            how = 'reflect';
        else % fxr >= fv(:,n) 
            % Perform contraction
            if fxr < fv(:,end)
                % Perform an outside contraction
                xc = (1 + psi*rho)*xbar - psi*rho*v(:,end);
                x(:) = xc; fxc = feval(funfcn,x,varargin{:});
                func_evals = func_evals+1;
                
                if fxc <= fxr
                    v(:,end) = xc; 
                    fv(:,end) = fxc;
                    how = 'contract outside';
                else
                    % perform a shrink
                    how = 'shrink'; 
                end
            else
                % Perform an inside contraction
                xcc = (1-psi)*xbar + psi*v(:,end);
                x(:) = xcc; fxcc = feval(funfcn,x,varargin{:});
                func_evals = func_evals+1;
                
                if fxcc < fv(:,end)
                    v(:,end) = xcc;
                    fv(:,end) = fxcc;
                    how = 'contract inside';
                else
                    % perform a shrink
                    how = 'shrink';
                end
            end
            if strcmp(how,'shrink')
                for j=two2np1
                    v(:,j)=v(:,1)+sigma*(v(:,j) - v(:,1));
                    x(:) = v(:,j); fv(:,j) = feval(funfcn,x,varargin{:});
                end
                func_evals = func_evals + n;
            end
        end
    end
    [fv,j] = sort(fv);
    v = v(:,j);
    itercount = itercount + 1;
    if prnt == 3
        fprintf(' %5.0f        %5.0f     %6.4e      %20s   ', itercount, func_evals, fv(1), how)
        fprintf('  x = [ ');for jj=1:length(x); fprintf(' %6.4e  ',x(jj));end;fprintf(' ]\n')
        
          save x x
        
    elseif prnt == 4
        disp(' ')
        disp(how)
        v
        fv
        func_evals
    end  
end   % while


x(:) = v(:,1);
if prnt == 4,
    % reset format
    set(0,{'format','formatspacing'},formatsave);
end
output.iterations = itercount;
output.funcCount = func_evals;
output.algorithm = 'Nelder-Mead simplex direct search';

fval = min(fv); 
if func_evals >= maxfun 
    if prnt > 0
        disp(' ')
        disp('Exiting: Maximum number of function evaluations has been exceeded')
        disp('         - increase MaxFunEvals option.')
        msg = sprintf('         Current function value: %f \n', fval);
        disp(msg)
    end
    exitflag = 0;
elseif itercount >= maxiter 
    if prnt > 0
        disp(' ')
        disp('Exiting: Maximum number of iterations has been exceeded')
        disp('         - increase MaxIter option.')
        msg = sprintf('         Current function value: %f \n', fval);
        disp(msg)
    end
    exitflag = 0; 
else
    if prnt > 1
        convmsg1 = sprintf([ ...
                '\nOptimization terminated successfully:\n',...
                ' the current x satisfies the termination criteria using OPTIONS.TolX of %e \n',...
                ' and F(X) satisfies the convergence criteria using OPTIONS.TolFun of %e \n'
        ],tolx, tolf);
        disp(convmsg1)
    end
    exitflag = 1;
end
